-- Ice Boat Racing Dynamic Soundtrack System
-- Plays different songs based on race position

-- Enable timing system scoreboard hook for races
hookTimingSystemScoreboardForRaces = true
runOutOfBoat = true

-- Initialize persistent variables
if not currentSong then
    currentSong = ""
end

if not currentSoundId then
    currentSoundId = -1
end

if not lastPosition then
    lastPosition = -1
end

if not raceStarted then
    raceStarted = false
end

if not inPitLast then
    inPitLast = false
end

-- Song definitions by position groups with generic names
local songGroups = {
    -- P1 (1st place) - Victory/Leading songs
    {
        "track_01",
        "track_02", 
        "track_03"
    },
    -- P2-P5 (Close competition) - Intense racing songs
    {
        "track_04",
        "track_05",
        "track_06"
    },
    -- P6-P10 (Mid-pack) - Determined/climbing songs
    {
        "track_07",
        "track_08",
        "track_09"
    },
    -- P11+ (Back of pack) - Underdog/comeback songs
    {
        "track_10",
        "track_11",
        "track_12"
    }
}

-- Function to get song group based on position
local function getSongGroup(position, inPit)
    if inPit then
        return 100
    end
    
    if position == 1 then
        return 1
    elseif position >= 2 and position <= 5 then
        return 2
    elseif position >= 6 and position <= 10 then
        return 3
    else
        return 4
    end
end

-- Function to get a random song from a group
local function getRandomSong(groupIndex)
    if groupIndex == 100 then
        return "in_pit"
    end
    
    local songs = songGroups[groupIndex]
    if songs and #songs > 0 then
        return songs[math.random(1, #songs)]
    end
    return ""
end

-- Main soundtrack logic
local currentPosition = nil
local playerFound = false
local totalPlayers = 0
local inPit = false

-- Check if we have timing system data
if timingSystemScoreboard and timingSystemScoreboard.rows and timingSystemScoreboard.extra.total_laps then
    -- First, check if this is valid race data by looking for any race-specific indicators
    for _, row in pairs(timingSystemScoreboard.rows) do
        totalPlayers = totalPlayers + 1
        
        -- Look for our player in the race data
        if row.player_name == playerName then
            playerFound = true
            
            -- Check multiple indicators that a race has started/is active
            if row.position or row.checkpoints or row.laps or row.time_diff_from_first_ms or row.in_pit ~= nil then
                raceStarted = true
                
                -- Get position if available, otherwise default to total players (last place)
                if row.position then
                    currentPosition = row.position
                else
                    -- If no position yet but race data exists, assume last place for now
                    currentPosition = totalPlayers
                end
                
                if row.in_pit then
                    inPit = true;
                end
            end
        end
    end
else
    -- No timing system data available
    raceStarted = false
end

-- Handle soundtrack changes
if raceStarted and playerFound and currentPosition then
    local targetSongGroup = getSongGroup(currentPosition, inPit)
    
    -- Check if we need to change song group
    if lastPosition == -1 or getSongGroup(lastPosition, inPitLast) ~= targetSongGroup then
        -- Stop current song if any
        if currentSoundId ~= -1 then
            stopSound(currentSoundId)
            currentSoundId = -1
        end
        
        -- Select and start new song
        currentSong = getRandomSong(targetSongGroup)
        if currentSong ~= "" then
            currentSoundId = playSound(currentSong, 1, 1)
        end
    end
    
    lastPosition = currentPosition
    inPitLast = inPit
else
    -- Stop music when not in a race
    if currentSoundId ~= -1 then
        stopSound(currentSoundId)
        currentSoundId = -1
    end
    
    currentSong = ""
    lastPosition = -1
    inPitLast = false
end

-- Display current soundtrack info
local yOffset = -90

-- Background panel - make it taller to accommodate more info
renderRect(300, 160, 0xAA000000, 'MIDDLE_CENTER')

offsetPosition(0, yOffset, function()
    renderText('🎵 DYNAMIC SOUNDTRACK', 0xFFFFD700, true, 'MIDDLE_CENTER')
end)

yOffset = yOffset + 25

-- Race status
offsetPosition(0, yOffset, function()
    local statusText = raceStarted and "🏁 RACE ACTIVE" or "⏸️ NO RACE"
    local statusColor = raceStarted and 0xFF00FF00 or 0xFFFF6600
    renderText(statusText, statusColor, true, 'MIDDLE_CENTER')
end)
yOffset = yOffset + 20

if raceStarted and playerFound and currentPosition then
    offsetPosition(0, yOffset, function()
        local positionText = "Position: " .. currentPosition .. "/" .. totalPlayers
        local positionColor = currentPosition == 1 and 0xFFFFD700 or 
                             (currentPosition <= 5 and 0xFF00FF00 or 0xFFFFFFFF)
        renderText(positionText, positionColor, true, 'MIDDLE_CENTER')
    end)
    yOffset = yOffset + 20
    
    local songGroup = getSongGroup(currentPosition, inPit)
    local groupNames = {"🥇 VICTORY", "🔥 INTENSE", "⚡ CLIMBING", "💪 COMEBACK"}
    if songGroup == 100 then
        groupName = "In Pit"
    else
        groupName = groupNames[songGroup]
    end
    
    offsetPosition(0, yOffset, function()
        renderText("Mode: " .. groupName, 0xFF00FFFF, true, 'MIDDLE_CENTER')
    end)
    yOffset = yOffset + 20
elseif raceStarted then
    offsetPosition(0, yOffset, function()
        renderText('In race, waiting for position...', 0xFFFFFF00, true, 'MIDDLE_CENTER')
    end)
    yOffset = yOffset + 40
else
    offsetPosition(0, yOffset, function()
        renderText('Waiting for race to start...', 0xFFFF6600, true, 'MIDDLE_CENTER')
    end)
    yOffset = yOffset + 40
end

if currentSong ~= "" then
    offsetPosition(0, yOffset, function()
        renderText("♪ NOW PLAYING:", 0xFF00FF00, true, 'MIDDLE_CENTER')
    end)
    yOffset = yOffset + 15
    
    offsetPosition(0, yOffset, function()
        renderText(currentSong, 0xFFFFFFFF, true, 'MIDDLE_CENTER')
    end)
    yOffset = yOffset + 20
    
    -- Show sound ID for debugging
    if currentSoundId ~= -1 then
        offsetPosition(0, yOffset, function()
            renderText("Sound ID: " .. currentSoundId, 0xFFAAAAAA, true, 'MIDDLE_CENTER')
        end)
        yOffset = yOffset + 15
    end
else
    offsetPosition(0, yOffset + 30, function()
        renderText("No song", 0xFFFF0000, true, 'MIDDLE_CENTER')
    end)
end

-- Audio system status
offsetPosition(0, yOffset, function()
    renderText('Audio System: ' .. (currentSong and 'ACTIVE' or 'STANDBY'), 
               currentSong and 0xFF00FF00 or 0xFFFF6600, true, 'MIDDLE_CENTER')
end)
